/*
 * Use Streams Pipes for communication between PGP clients
 * and pgpsdkd.
 *
 * Reference:
 *  UNIX System V Network Programming
 *  Stephen A. Rago
 *  Section 3.6 - IPC WITH STREAMS PIPES
 */

#include <stdio.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <pthread.h>
#include <sys/stropts.h>
#include <sys/resource.h>
#include <signal.h>

#include <pgpUtilities.h>
#include <pgpThreads.h>
#include <pgpRPCMsg.h>
#include "pgpPFLErrors.h"

extern int debug;
PGPMutex_t sRPCMutex;
void * conn_thread(void *);
int pfd[2];
int numthreads = 0;
int maxthreads = 0;
PGPMutex_t threadmutex;

#define PIPEPATH "/var/.pgpsdkpipe"
#define PIPEMODE 0766

struct argpack {
	int fd, uid, gid;
};

uid_t	rootUserID;
gid_t	rootGrpID;

run_pgpsdkd()
{
	int sd, n, fd;
	int rv = 0;
	struct rlimit	limits;
	pthread_t thread;
	pthread_attr_t attr;

	pthread_attr_init(&attr);
	pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);

	rootUserID = geteuid();
	rootGrpID = getegid();

	/*	Don't barf on SIGPIPE 	*/
	signal(SIGPIPE, SIG_IGN);
	/*
	 *	Set max num of file descriptors.
 	 */
	rv = getrlimit(RLIMIT_NOFILE, &limits);
	if(rv)
	{
		perror("getrlimit()");
		exit(1);
	}
	else
	{
		limits.rlim_cur = limits.rlim_max;
		rv = setrlimit(RLIMIT_NOFILE, &limits);
		if(rv)
		{
			perror("setrlimit()");
			exit(1);
		}
		rv = getrlimit(RLIMIT_NOFILE, &limits);
		maxthreads = limits.rlim_max / 4;
		if(maxthreads == 0)
		{
			perror("getrlimit(2)");
			exit(1);
		}
	}
	/*
	 *	Create mutex used to keep track of number of threads running
	 */
	PGPMutexCreate(&threadmutex, NULL);

	if (getuid() != 0) {
		printf("NOTICE: pgpsdkd not running as root\n");
	}

	PGPMutexCreate(&sRPCMutex, NULL);

	pgpRPCCreateBEContext();	/* does PGPsdkInit() too..  */

	if (pipe(pfd) < 0) {
		perror("pipe()");
		exit(1);
	}

	/*
	 * Push CONNLD on one end to enable unique
	 * connections.
	 */
	if (ioctl(pfd[1], I_PUSH, "connld") < 0) {
		fprintf(stderr, "cannot push CONNLD");
		exit(1);
	}

	unlink(PIPEPATH);
	fd = creat(PIPEPATH, PIPEMODE);
	if (fd == -1) {
		char buf[80];
		sprintf(buf, "create(%s)", PIPEPATH);
		perror(buf);
		exit(1);
	}
	close(fd);
	chmod(PIPEPATH, PIPEMODE);

	if (fattach(pfd[1], PIPEPATH) < 0) {
		perror("fattach()");
		exit(1);
	}

	if (debug)
		printf("WAITING FOR CONNECTIONS\n");

	for(;;) {
		struct strrecvfd recvfd;
		struct argpack *argp;

		if (ioctl(pfd[0], I_RECVFD, &recvfd) < 0)
			continue;

		argp = (struct argpack *)malloc(sizeof(*argp));
		argp->fd = recvfd.fd;
		argp->uid = recvfd.uid;
		argp->gid = recvfd.gid;

		while(numthreads >= maxthreads)
		{
			sleep(1);
		}
		rv = pthread_create(&thread, &attr, conn_thread, argp);
		if(rv != 0)
		{
			perror("pthread_create()");
			exit(1);
		}
		else
		{
			PGPMutexLock(&threadmutex);
			numthreads++;
			PGPMutexUnlock(&threadmutex);
		}
	}
}

void *
conn_thread(void *arg)
{
	struct argpack *argp = (struct argpack *)arg;
	int			n, sd = argp->fd;
	char			*buf = NULL;
	char			*next = NULL;
	PGPPackMsg		pkt;
	PGPInt32		length = 0;
	PGPInt32		currentSize = 0;
	const PGPInt32		headerSize = sizeof(PGPInt32) * 3;
	PGPContextRef		context = kInvalidPGPContextRef;
	char			*outbuf = NULL;
	int			outbuflen;
	char			*bufcopy = NULL;
	int			uid = argp->uid;
	int			gid = argp->gid;

	free((struct argpack *)arg);

	if(PGPNewContext(kPGPsdkAPIVersion, &context) != kPGPError_NoErr)
		goto out;
	buf = (char *)PGPNewSecureData(PGPPeekContextMemoryMgr(context), headerSize, 0);
	if(buf == NULL)
		goto out;
	currentSize = headerSize;

	for(;;)
	{
		/*	Read header	*/
		n = read(sd, buf, headerSize);
		if (n == 0)
		{
			if (debug)
				printf("connection severed: thread exiting\n");
			goto out;		/* Exit Thread */
		}
		if (n == -1)
		{
			perror("read");
			goto out;		/* Exit Thread */
		}
		pkt.ptr = 0;
		pkt.length = 0;
		pkt.base = buf;

		/* 	skip first two ints in header	*/
		unpack_int32(&pkt);
		unpack_int32(&pkt);
		length = unpack_int32(&pkt);

		if(length != 0)
		{
			if(length  > currentSize)
			{
				currentSize = length;
				if(PGPReallocData(PGPPeekContextMemoryMgr(context), (void **)&buf, currentSize, 0) != kPGPError_NoErr)
					goto out;
			}
		}
		else
			goto out;
		if(length > headerSize)
		{
			next = buf;
			next += headerSize;
			n += read(sd, next, length - headerSize);

			if(n == 0 || n == -1)
				goto out;
			while(n != length)
			{
				/*	Read remaining data 	*/
				next = buf + n;
				n += read(sd, next, length - n);
			}
		}

		PGPMutexLock(&sRPCMutex);
		
		if(setegid(gid) == -1)
		{
			PGPMutexUnlock(&sRPCMutex);
			goto out;
		}
		if(seteuid(uid) == -1)
		{
			PGPMutexUnlock(&sRPCMutex);
			goto out;
		}

		rcv_generic(buf, n, (PGPByte **)&outbuf, &outbuflen);

		if(outbuflen < currentSize)
		{
			memcpy(buf, outbuf, outbuflen);
		}
		else
		{
			currentSize = outbuflen;
			if(PGPReallocData(PGPPeekContextMemoryMgr(context), (void **)&buf, currentSize, 0) != kPGPError_NoErr)
			{
				PGPMutexUnlock(&sRPCMutex);
				goto out;
			}
			memcpy(buf, outbuf, outbuflen);
		}

		if (!PGPIsInitRPCBuf(outbuf))
			PGPFreeData(outbuf);

		if(seteuid(rootUserID) == -1)
		{
			PGPMutexUnlock(&sRPCMutex);
			goto out;
		}
		if(setegid(rootGrpID) == -1)
		{
			PGPMutexUnlock(&sRPCMutex);
			goto out;
		}

		PGPMutexUnlock(&sRPCMutex);

		n = write(sd, buf, outbuflen);
		if (n != outbuflen)
		{
			goto out;
		}
	}
out:	
	if(buf != NULL)
		PGPFreeData(buf);
	PGPMutexLock(&sRPCMutex);
	setegid(gid);
	seteuid(uid);
	if(context != kInvalidPGPContextRef)
		PGPFreeContext(context);
	seteuid(rootUserID);
	setegid(rootGrpID);
	PGPMutexUnlock(&sRPCMutex);
	close(sd);
	PGPMutexLock(&threadmutex);
	numthreads--;
	PGPMutexUnlock(&threadmutex);
}

void
pgpRPCGetUnixClientAuth(pgpRPCconnection *conn_ref)
{
	char name[20];
	PGPByte *cp;

	sprintf(name, "%d", geteuid());
	cp = PGPNewData(PGPGetDefaultMemoryMgr(), strlen(name)+1,0);
	pgpCopyMemory(name, cp, strlen(name)+1);
	conn_ref->UserName = cp;
	conn_ref->uid = -1;  /* Only used with Doors API */
	conn_ref->gid = -1;  /* Only used with Doors API */
	conn_ref->pid = -1;  /* Only used with Doors API */
}

PGPBoolean
pgpRPCVerifyUnixClientAuth(pgpRPCconnection *conn_ref)
{
	return TRUE;
}

